import bpy
import math
import os
from mathutils import Vector
from bpy.props import (
    FloatProperty, EnumProperty, IntProperty, StringProperty,
    PointerProperty, BoolProperty, CollectionProperty
)

# -------------------------------
# PROPIEDADES
# -------------------------------
class CNCTabItem(bpy.types.PropertyGroup):
    x: bpy.props.FloatProperty(name="X")
    y: bpy.props.FloatProperty(name="Y")

class CNCJobItem(bpy.types.PropertyGroup):
    name: StringProperty(name="Nombre", default="Corte")
    object_names: StringProperty(name="Objects", default="")
    diameter: FloatProperty(name="Diámetro", default=3.0)
    comp: EnumProperty(
        name="Compensación",
        items=[('CENTER','Centro',''),('Interior','Interior',''),('Exterior','Exterior','')],
        default='CENTER'
    )
    depth_final: FloatProperty(name="Final", default=2.0)
    depth_per_pass: FloatProperty(name="Pasada", default=2.0)
    feed: FloatProperty(name="De corte", default=900.0)
    plunge: FloatProperty(name="Bajada", default=600.0)
    z_safe: FloatProperty(name="Z safe", default=5.0)
    sample_steps: IntProperty(name="Muestreo", default=200)

    # pestañas
    tabs_enable: BoolProperty(name="Usar pestañas (mm)", default=False)
    tab_length: FloatProperty(name="Longitud", default=3.0, min=0.1)
    tab_thickness: FloatProperty(name="Espesor", default=2, min=0.0)
    tab_count: IntProperty(name="Cantidad", default=3, min=0, max=20)

    tab_mode: EnumProperty(
        name="Modo pestañas",
        items=[
            ('AUTO', "Automático", "Generar pestañas distribuidas"),
            ('MANUAL', "Manual", "Ir a modo edición para seleccionar puntos"),
        ],
        default='AUTO'
    )

    custom_tabs: CollectionProperty(type=CNCTabItem)

    cut_direction: EnumProperty(
        name="Dirección de corte",
        description="Sentido del movimiento de la herramienta",
        items=[
            ('CLOCKWISE', "Horario", "Corte en sentido horario"),
            ('COUNTERCLOCKWISE', "Antihorario (Convencional)", "Corte en sentido antihorario"),
        ],
        default='COUNTERCLOCKWISE'
    )

class CNCProperties(bpy.types.PropertyGroup):
    diameter: FloatProperty(name="Diámetro fresa (mm)", default=3.0, min=0.1)
    depth_final: FloatProperty(name="Profundidad final", default=2.0, min=0.01)
    depth_per_pass: FloatProperty(name="Profundidad por pasada", default=2.0, min=0.01)
    feed: FloatProperty(name="Feed (mm/min)", default=900.0, min=1.0)
    plunge: FloatProperty(name="Plunge (mm/min)", default=600.0, min=1.0)
    z_safe: FloatProperty(name="Z seguro", default=5.0)

    comp: EnumProperty(
        name="Compensación para G-code",
        description="Tipo de compensación aplicada al generar G-code",
        items=[
            ('CENTER', 'Centro', 'Sin desplazamiento'),
            ('Interior', 'Interior', 'Compensado hacia adentro'),
            ('Exterior', 'Exterior', 'Compensado hacia afuera')
        ],
        default='CENTER'
    )

    contour_generation_mode: EnumProperty(
        name="Tipo de contorno",
        description="Crear",
        items=[
            ('CENTER', "Sobre la línea", "Sin compensación"),
            ('Interior', "Interior", "Por dentro de la línea"),
            ('Exterior', "Exterior", "Por fuera de la línea")
        ],
        default='CENTER'
    )

    sample_steps: IntProperty(name="Muestreo curva", default=200, min=10, max=2000)
    file_path: StringProperty(name="Archivo salida", default="", subtype='FILE_PATH')
    auto_clear_paths: BoolProperty(name="Limpiar colección antes", default=True)
    show_additional_info: BoolProperty(name="Ver info adicional", default=False)

    contour_coord_mode: EnumProperty(
        name="Mostrar coordenadas",
        description="Selecciona qué coordenadas mostrar",
        items=[
            ('NONE', "Ninguno", ""),
            ('CENTER', "Centro", ""),
            ('MIN', "Esquina inferior izquierda", ""),
            ('BOTH', "Centro y Esquina", "")
        ],
        default='NONE'
    )

    jobs: CollectionProperty(type=CNCJobItem)
    jobs_index: IntProperty(name="Job index", default=0)

classes = (CNCTabItem, CNCJobItem, CNCProperties)

# -------------------------------
# FUNCIONES DE GEOMETRÍA Y COMPENSADO
# -------------------------------

def clean_polyline(poly, tolerance=1e-5):
    """
    Limpia duplicados consecutivos y asegura cierre limpio.
    """
    if len(poly) < 2:
        return poly
    cleaned = [poly[0]]
    for i in range(1, len(poly)):
        prev = Vector(cleaned[-1])
        curr = Vector(poly[i])
        if (curr - prev).length > tolerance:
            cleaned.append(poly[i])
    if len(cleaned) > 2:
        if (Vector(cleaned[0]) - Vector(cleaned[-1])).length < tolerance:
            cleaned.pop()
    return cleaned

def is_clockwise(poly):
    """
    Determina si un polígono es horario (CW) usando el área con signo.
    """
    poly_clean = clean_polyline(poly, tolerance=1e-4)
    if len(poly_clean) < 3:
        return False
    area = 0.0
    for i in range(len(poly_clean)):
        x1, y1 = poly_clean[i]
        x2, y2 = poly_clean[(i + 1) % len(poly_clean)]
        area += (x1 * y2) - (x2 * y1)
    EPSILON = 1e-4
    return area < -EPSILON  # Negativo = CW

def normalize_poly_orientation(poly, clockwise=False):
    """
    Fuerza orientación CCW por defecto.
    """
    is_cw = is_clockwise(poly)
    if is_cw != clockwise:
        return list(reversed(poly))
    return poly

def apply_cut_direction(poly, direction_enum):
    """
    Aplica la dirección de corte final (Horario vs Antihorario).
    """
    target_cw = (direction_enum == 'CLOCKWISE')
    current_cw = is_clockwise(poly)
    if target_cw != current_cw:
        return list(reversed(poly))
    return poly

def simplificar_polilinea(poly, tolerancia=0.05):
    """
    Simplifica polilínea eliminando puntos muy cercanos.
    """
    if len(poly) < 4:
        return poly[:]
    simple = [poly[0]]
    for i in range(1, len(poly)-1):
        if (Vector(poly[i]) - Vector(simple[-1])).length > tolerancia:
            simple.append(poly[i])
    simple.append(poly[-1])
    if len(simple) < 3 and len(poly) >= 3:
        return poly[:3] + [poly[0]]
    return simple

def offset_polyline(poly, offset):
    """
    Offset robusto usando Shapely si está disponible.
    Asume que la poly de entrada es CCW.
    """
    if len(poly) < 3:
        return poly[:]
    poly_simple = simplificar_polilinea(poly, tolerancia=0.05)
    try:
        from shapely.geometry import Polygon
        polygon = Polygon(poly_simple)
        offset_polygon = polygon.buffer(
            offset,
            join_style=2,  # mitre
            mitre_limit=5.0,
            resolution=16
        )
        if offset_polygon.is_empty:
            return []
        if hasattr(offset_polygon, 'geoms'):
            coords = list(offset_polygon.geoms[0].exterior.coords)
        else:
            coords = list(offset_polygon.exterior.coords)
        if coords and (Vector(coords[0]) - Vector(coords[-1])).length > 1e-6:
            coords.append(coords[0])
        return coords
    except Exception:
        return offset_polyline_fallback(poly_simple, offset)

def offset_polyline_fallback(poly, offset):
    """
    Algoritmo propio robusto como fallback.
    Asume poly CCW de entrada.
    """
    if len(poly) < 3:
        return poly[:]
    pts = [Vector(p) for p in poly]
    closed_pts = pts[:]
    if (pts[0] - pts[-1]).length > 1e-6:
        closed_pts.append(pts[0])
    offset_poly = []
    count = len(closed_pts) - 1
    for i in range(count):
        p_prev = closed_pts[(i - 1) % count]
        p = closed_pts[i]
        p_next = closed_pts[(i + 1) % count]
        v1 = p - p_prev
        v2 = p_next - p
        valid_vectors = []
        if v1.length > 1e-6: valid_vectors.append(v1.normalized())
        if v2.length > 1e-6: valid_vectors.append(v2.normalized())
        if not valid_vectors:
            offset_point = p
        elif len(valid_vectors) == 1:
            v_norm = valid_vectors[0]
            normal = Vector((v_norm.y, -v_norm.x))  # normal derecha
            offset_point = p + normal * offset
        else:
            v1n, v2n = valid_vectors
            n1 = Vector((v1n.y, -v1n.x))
            n2 = Vector((v2n.y, -v2n.x))
            bisector = n1 + n2
            dot = v1n.dot(v2n)
            dot = max(min(dot, 1.0), -1.0)
            if bisector.length > 1e-3:
                bisector.normalize()
                cos_half = n1.dot(bisector)
                if abs(cos_half) > 1e-4:
                    scale = 1.0 / cos_half
                    scale = min(scale, 5.0)
                    offset_point = p + bisector * (offset * scale)
                else:
                    offset_point = p + bisector * offset
            else:
                offset_point = p + n1 * offset
        offset_poly.append((offset_point.x, offset_point.y))
    if offset_poly and (Vector(offset_poly[0]) - Vector(offset_poly[-1])).length > 1e-6:
        offset_poly.append(offset_poly[0])
    return offset_poly

def get_offset_for_compensation(job):
    """
    Devuelve el valor de offset según la compensación elegida.
    Interior = negativo, Exterior = positivo, CENTER = 0.
    """
    if job.comp == 'Interior':
        return -job.diameter / 2.0
    elif job.comp == 'Exterior':
        return job.diameter / 2.0
    else:
        return 0.0

def compensated_poly(poly, job):
    """
    Aplica compensación robusta:
    - Normaliza orientación CCW
    - Aplica offset según Interior/Exterior
    - Aplica dirección de corte
    """
    poly = normalize_poly_orientation(poly, clockwise=False)
    offset_value = get_offset_for_compensation(job)
    if abs(offset_value) > 1e-6:
        new_poly = offset_polyline(poly, offset_value)
        if new_poly and len(new_poly) > 2:
            poly = new_poly
    poly = apply_cut_direction(poly, job.cut_direction)
    return poly

# -------------------------------
# CREACIÓN DE CURVAS Y PESTAÑAS
# -------------------------------

def sample_curve_object_to_xy_list(obj, sample_steps=200):
    """
    Extrae polígonos XY respetando el orden de vértices de un objeto curva.
    """
    coords = []
    if obj.type != 'CURVE':
        return []

    deps = bpy.context.evaluated_depsgraph_get()
    eval_obj = obj.evaluated_get(deps)

    try:
        me = eval_obj.to_mesh()
    except Exception:
        return []
    if me is None:
        return []

    mat_world = obj.matrix_world.copy()
    verts_world = [mat_world @ Vector(v.co) for v in me.vertices]
    edges = [(e.vertices[0], e.vertices[1]) for e in me.edges]

    adj = {}
    for a, b in edges:
        adj.setdefault(a, []).append(b)
        adj.setdefault(b, []).append(a)

    visited = set()
    for vi in range(len(verts_world)):
        if vi in visited or vi not in adj:
            continue
        path_indices = [vi]
        visited.add(vi)
        curr = vi
        while True:
            neighbors = adj.get(curr, [])
            next_v = None
            for n in neighbors:
                if n not in visited:
                    next_v = n
                    break
            if next_v is None:
                if path_indices[0] in neighbors and len(path_indices) > 2:
                    break
                else:
                    break
            visited.add(next_v)
            path_indices.append(next_v)
            curr = next_v
        poly = [(verts_world[i].x, verts_world[i].y) for i in path_indices]
        poly = clean_polyline(poly)
        if len(poly) >= 3:
            if (Vector(poly[0]) - Vector(poly[-1])).length > 1e-4:
                poly.append(poly[0])
            coords.append(poly)

    try:
        eval_obj.to_mesh_clear()
    except Exception:
        pass
    return coords




def create_offset_curve_object(name, poly_points, ref_obj=None):
    """
    Crear un objeto CURVE en Blender a partir de una lista de puntos compensados.
    poly_points debe ser lista de (x,y).
    """

    curve_data = bpy.data.curves.new(name=name, type='CURVE')
    curve_data.dimensions = '2D'
    curve_data.fill_mode = 'NONE'

    spline = curve_data.splines.new('POLY')
    spline.points.add(len(poly_points) - 1)

    for i, (x, y) in enumerate(poly_points):
        if ref_obj:
            # ⚡ Transformar cada punto a coordenadas globales
            world_co = ref_obj.matrix_world @ Vector((x, y, 0.0))
            spline.points[i].co = (world_co.x, world_co.y, world_co.z, 1.0)
        else:
            spline.points[i].co = (x, y, 0.0, 1.0)

    new_obj = bpy.data.objects.new(name, curve_data)

    # Si hay objeto de referencia, copiar ubicación
    if ref_obj:
        new_obj.location = ref_obj.location

    bpy.context.scene.collection.objects.link(new_obj)
    return new_obj

def compute_tab_ranges_on_poly(poly, tab_length, tab_count):
    """
    Calcula rangos de pestañas distribuidos uniformemente en el perímetro.
    """
    if len(poly) < 3:
        return []
    cum = [0.0]
    for i in range(len(poly) - 1):
        cum.append(cum[-1] + (Vector(poly[i+1]) - Vector(poly[i])).length)
    perimeter = cum[-1]
    if perimeter <= 1e-6 or tab_count <= 0:
        return []
    centers = [perimeter * (k + 0.5) / tab_count for k in range(tab_count)]
    ranges = []
    half = tab_length / 2.0
    for c in centers:
        s, e = c - half, c + half
        if s < 0:
            ranges.append((0.0, e))
            ranges.append((perimeter + s, perimeter))
        elif e > perimeter:
            ranges.append((s, perimeter))
            ranges.append((0.0, e - perimeter))
        else:
            ranges.append((s, e))
    return ranges

def point_distance_along_poly(poly):
    """
    Devuelve lista de distancias acumuladas a lo largo de la polilínea.
    """
    dists = [0.0]
    for i in range(len(poly) - 1):
        dists.append(dists[-1] + (Vector(poly[i+1]) - Vector(poly[i])).length)
    return dists

def path_length(poly):
    """
    Calcula la longitud total de una polilínea.
    """
    d = 0.0
    for i in range(len(poly) - 1):
        a = Vector(poly[i])
        b = Vector(poly[i + 1])
        d += (b - a).length
    return d

def compute_selected_contour_size(context, sample_steps=200):
    """
    Calcula tamaño, perímetro y centro del primer contorno seleccionado.
    """
    sel = [o for o in context.selected_objects if o.type == 'CURVE']
    if not sel:
        return None
    obj = sel[0]
    polys = sample_curve_object_to_xy_list(obj, sample_steps=sample_steps)
    if not polys:
        return None
    min_x = float('inf')
    min_y = float('inf')
    max_x = -float('inf')
    max_y = -float('inf')
    total_perimeter = 0.0
    total_verts = 0
    for poly in polys:
        for (x, y) in poly:
            min_x = min(min_x, x)
            min_y = min(min_y, y)
            max_x = max(max_x, x)
            max_y = max(max_y, y)
            total_verts += 1
        total_perimeter += path_length(poly)
    width = max_x - min_x
    height = max_y - min_y
    center_x = (min_x + max_x) / 2.0
    center_y = (min_y + max_y) / 2.0
    return {
        'object_name': obj.name,
        'width': width,
        'height': height,
        'perimeter': total_perimeter,
        'bounds_min': (min_x, min_y),
        'bounds_max': (max_x, max_y),
        'center': (center_x, center_y),
        'polys_count': len(polys),
        'verts_count': total_verts
    }


# -------------------------------
# GENERACIÓN DE G-CODE
# -------------------------------

def generate_gcode_for_poly(poly, job, per_path_tab_ranges):
    """
    Genera G-code optimizado:
    - Normaliza orientación CCW
    - Aplica compensación Interior/Exterior robusta
    - Aplica dirección de corte
    - Genera pasadas con pestañas en la última
    """

    # 1. Aplicar compensación robusta
    poly = compensated_poly(poly, job)

    # -------------------------------
    # G-code
    # -------------------------------
    lines = []
    z_safe = job.z_safe
    depth_final = job.depth_final
    depth_per_pass = job.depth_per_pass
    feed = job.feed
    plunge = job.plunge

    # Calcular profundidades
    depths = []
    current = 0.0
    while current > -depth_final:
        current -= depth_per_pass
        if current < -depth_final:
            current = -depth_final
        depths.append(current)
    if not depths:
        depths.append(-depth_final)

    # Punto inicial
    start_x, start_y = poly[0]
    lines.append(f"G0 Z{z_safe:.3f}")
    lines.append(f"G0 X{start_x:.3f} Y{start_y:.3f}")

    # Calcular distancias para pestañas
    dists = point_distance_along_poly(poly)
    tab_ranges = per_path_tab_ranges[0] if per_path_tab_ranges else []

    def write_poly_lines(z_target, use_tabs=False):
        lines.append(f"G1 Z{z_target:.3f} F{plunge:.1f}")
        last_x, last_y = start_x, start_y
        Interior_tab = False
        for i in range(1, len(poly)):
            x, y = poly[i]
            prev_x, prev_y = poly[i-1]
            if use_tabs and tab_ranges and dists:
                d_prev = dists[i-1]
                d_curr = dists[i]
                events = []
                for ts, te in tab_ranges:
                    if d_prev <= ts <= d_curr:
                        t = (ts - d_prev) / (d_curr - d_prev + 1e-9)
                        events.append((ts, prev_x + (x-prev_x)*t, prev_y + (y-prev_y)*t, 'enter'))
                    if d_prev <= te <= d_curr:
                        t = (te - d_prev) / (d_curr - d_prev + 1e-9)
                        events.append((te, prev_x + (x-prev_x)*t, prev_y + (y-prev_y)*t, 'exit'))
                if events:
                    events.sort(key=lambda e: e[0])
                    for _, ex, ey, etype in events:
                        lines.append(f"G1 X{ex:.3f} Y{ey:.3f} F{feed:.1f}")
                        if etype == 'enter' and not Interior_tab:
                            lines.append(f"G1 Z{z_target + job.tab_thickness:.3f} F{plunge:.1f}")
                            Interior_tab = True
                        elif etype == 'exit' and Interior_tab:
                            lines.append(f"G1 Z{z_target:.3f} F{plunge:.1f}")
                            Interior_tab = False
            lines.append(f"G1 X{x:.3f} Y{y:.3f} F{feed:.1f}")
            last_x, last_y = x, y
        # Cerrar polígono
        if (Vector((last_x, last_y)) - Vector((start_x, start_y))).length > 1e-3:
            lines.append(f"G1 X{start_x:.3f} Y{start_y:.3f} F{feed:.1f}")
        if Interior_tab:
            lines.append(f"G1 Z{z_target:.3f} F{plunge:.1f}")

    # Pasadas
    for idx, d in enumerate(depths):
        is_last = (idx == len(depths) - 1)
        lines.append(f"(Pasada Z{d:.3f})")
        write_poly_lines(d, use_tabs=(is_last and job.tabs_enable))

    # Retracción final
    lines.append(f"G0 Z{z_safe:.3f}")
    return lines
