import bpy
from . import utils

# --- UIList para mostrar cortes en la cola ---
class CNC_UL_jobs(bpy.types.UIList):
    def draw_item(self, context, layout, data, item, icon, active_data, active_propname, index):
        job = item
        if self.layout_type in {'DEFAULT', 'COMPACT'}:
            layout.label(text=job.name, icon='SEQ_STRIP_META')
            layout.label(text=job.object_names)
        elif self.layout_type in {'GRID'}:
            layout.alignment = 'CENTER'
            layout.label(text=job.name)




class OT_SelectAllJobObjects(bpy.types.Operator):
    bl_idname = "cnc.select_all_job_objects"
    bl_label = "Seleccionar todos los objetos del Job"
    bl_options = {'REGISTER', 'UNDO'}

    object_names: bpy.props.StringProperty(name="Nombres objetos")

    def execute(self, context):
        for o in context.view_layer.objects:
            o.select_set(False)
        found = 0
        for name in [n.strip() for n in self.object_names.split(",") if n.strip()]:
            obj = context.scene.objects.get(name)
            if obj:
                obj.select_set(True)
                found += 1
        if found:
            sel = context.selected_objects
            if sel:
                context.view_layer.objects.active = sel[0]
                try:
                    bpy.ops.view3d.view_selected(use_all_regions=False)
                except:
                    pass
            self.report({'INFO'}, f"Seleccionados {found} objeto(s)")
            return {'FINISHED'}
        else:
            self.report({'WARNING'}, "No se encontró ninguno de los objetos")
            return {'CANCELLED'}


# --- Panel principal ---
class VIEW3D_PT_CNCPanel(bpy.types.Panel):
    bl_label = "Gcode para Corte"
    bl_idname = "VIEW3D_PT_cnc_gcode_jobs_tabs"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = "Corte GCode"

    def draw(self, context):
        layout = self.layout
        props = context.scene.cnc_props

        # --- Sección GLOBAL ---
        layout.label(text="Parámetros globales")
        col = layout.column(align=True)
        col.prop(props, "diameter")
        col.prop(props, "z_safe")
        col.prop(props, "sample_steps")
        layout.prop(props, "auto_clear_paths")

        layout.separator()
        layout.label(text="Crear contornos compensados según la herramienta:")
        row = layout.row(align=True)
        row.prop(props, "contour_generation_mode", text="")
        row.operator("cnc.visualize_selected_compensation", icon='CURVE_PATH')

        # --- Medidas del contorno seleccionado ---
        sel_info = utils.compute_selected_contour_size(context, sample_steps=props.sample_steps)
        if sel_info:
            layout.separator()
            row = layout.row(align=True)
            row.label(text="Dimensiones:")
            
            row.label(text=f"Ancho X: {sel_info['width']:.3f} mm")
            row.label(text=f"Alto Y: {sel_info['height']:.3f} mm")

            layout.separator()
            layout.prop(props, "contour_coord_mode", text="Coordenadas")
            mode = props.contour_coord_mode
            if mode in ('CENTER', 'BOTH'):
                cx, cy = sel_info['center']
                row = layout.row(align=True)
                row.label(text=f"Centro X: {cx:.3f} mm")
                row.label(text=f"Centro Y: {cy:.3f} mm")
            if mode in ('MIN', 'BOTH'):
                mx, my = sel_info['bounds_min']
                row = layout.row(align=True)
                row.label(text=f"Esquina min X: {mx:.3f} mm")
                row.label(text=f"Esquina min Y: {my:.3f} mm")

            if props.show_additional_info:
                layout.separator()
                layout.label(text="Info adicional")
                row = layout.row(align=True)
                row.label(text=f"Perímetro: {sel_info['perimeter']:.3f} mm")
                row = layout.row(align=True)
                row.label(text=f"Polilíneas: {sel_info['polys_count']}")
                row.label(text=f"Vértices: {sel_info['verts_count']}")

        # --- Gestión de Jobs ---
        layout.separator()
        row = layout.row()
        row.operator("cnc.add_job", icon='ADD')
        row.operator("cnc.remove_job", icon='REMOVE')
        row.operator("cnc.clear_jobs", icon='TRASH')

        layout.template_list("CNC_UL_jobs", "", props, "jobs", props, "jobs_index", rows=4)

        # --- Sección JOB seleccionado ---
        if len(props.jobs) > 0:
            job = props.jobs[props.jobs_index]
            layout.separator()
            
         
            row = layout.row(align=True)
            row.label(text="Orden de cortes:")
            row.operator("cnc.move_job_up", text="Subir")
            row.operator("cnc.move_job_down", text="Bajar")

           # layout.label(text="Editar corte seleccionado")
            
            

            # Botones clicables para selección de objetos
            
            row = layout.row(align=True)
            row.prop(job, "name")
            op_all = row.operator("cnc.select_all_job_objects", text="Seleccionar corte", icon='CHECKBOX_HLT')
            op_all.object_names = job.object_names

          

            layout.separator()
            layout.label(text="Profundidad (mm)")
            row = layout.row(align=True)
            row.prop(job, "depth_per_pass")
            row.prop(job, "depth_final")
            layout.label(text="Velocidad (mm/min)")
            row = layout.row(align=True)
            row.prop(job, "feed")
            row.prop(job, "plunge")
            layout.prop(job, "cut_direction", text="Dirección")

            # --- Pestañas ---
            layout.separator()
            layout.prop(job, "tabs_enable", text="Usar pestañas")

            if job.tabs_enable:
                row = layout.row(align=True)
                row.prop(job, "tab_mode", text="Modo pestañas")

                row = layout.row(align=True)
                row.prop(job, "tab_length")
                row.prop(job, "tab_thickness")

                if job.tab_mode == 'AUTO':
                    layout.prop(job, "tab_count")
                elif job.tab_mode == 'MANUAL':
                    layout.label(text=f"Pestañas seleccionadas: {len(job.custom_tabs)}")
                    for i, tab in enumerate(job.custom_tabs):
                        layout.label(text=f"{i+1}: X={tab.x:.2f}, Y={tab.y:.2f}")

                    
                    layout.operator("cnc.add_custom_tab", icon='DOT')
                    layout.operator("cnc.reset_custom_tabs", icon='TRASH')
                layout.operator("cnc.visualize_tabs", icon='DOT')

        # --- Ruta de salida y exportación ---
        layout.separator()
        row = layout.row(align=True)
        row.prop(props, "file_path", text="Archivo salida")
        row.operator("cnc.select_output_path", text="", icon='FILE_FOLDER')

        layout.separator()
        layout.operator("cnc.export_jobs", icon='EXPORT')


# --- Registro ---
classes = (
    CNC_UL_jobs,
    VIEW3D_PT_CNCPanel,
    OT_SelectAllJobObjects,
)
