import bpy
import bmesh
from mathutils import Vector
import math
from . import utils

# OPERADOR: Seleccionar ruta de salida
class OT_SelectOutputPath(bpy.types.Operator):
    bl_idname = "cnc.select_output_path"
    bl_label = "Elegir ruta salida"
    bl_options = {'REGISTER', 'UNDO'}
    filepath: bpy.props.StringProperty(subtype="FILE_PATH", default="")

    def execute(self, context):
        context.scene.cnc_props.file_path = self.filepath
        self.report({'INFO'}, f"Ruta seleccionada: {self.filepath}")
        return {'FINISHED'}

    def invoke(self, context, event):
        if not self.filepath:
            self.filepath = bpy.path.abspath("//output.gcode")
        context.window_manager.fileselect_add(self)
        return {'RUNNING_MODAL'}




def create_offset_curve_object(name, poly_points, ref_obj=None):
    curve_data = bpy.data.curves.new(name=name, type='CURVE')
    curve_data.dimensions = '2D'
    curve_data.fill_mode = 'NONE'

    spline = curve_data.splines.new('POLY')
    spline.points.add(len(poly_points) - 1)

    for i, (x, y) in enumerate(poly_points):
        spline.points[i].co = (x, y, 0.0, 1.0)

    new_obj = bpy.data.objects.new(name, curve_data)
    bpy.context.scene.collection.objects.link(new_obj)

    if ref_obj:
        new_obj.matrix_world = ref_obj.matrix_world.copy()

    # ⚡ Asegurar que el objeto activo sea el nuevo
    bpy.ops.object.select_all(action='DESELECT')
    bpy.context.view_layer.objects.active = new_obj
    new_obj.select_set(True)
    bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='BOUNDS')
    new_obj.select_set(False)

    return new_obj


class OT_VisualizeSelectedCompensation(bpy.types.Operator):
    bl_idname = "cnc.visualize_selected_compensation"
    bl_label = "Crear:"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.cnc_props
        sel = [o for o in context.selected_objects if o.type == 'CURVE']
        if not sel:
            self.report({'WARNING'}, "Selecciona al menos una curva")
            return {'CANCELLED'}

        # Mapa RGB para cada tipo de compensación
        color_map = {
            'CENTER': (1.0, 1.0, 1.0),   # blanco
            'Interior': (1.0, 0.2, 0.2),   # rojo
            'Exterior': (0.2, 1.0, 0.2)   # verde
        }

        def ensure_material_with_color(name, rgb):
            mat = bpy.data.materials.get(name)
            if mat is None:
                mat = bpy.data.materials.new(name=name)
            mat.use_nodes = True
            bsdf = mat.node_tree.nodes.get("Principled BSDF")
            if bsdf:
                bsdf.inputs["Base Color"].default_value = (rgb[0], rgb[1], rgb[2], 1.0)
            return mat

        created = 0
        for obj in sel:
            polys = utils.sample_curve_object_to_xy_list(obj, sample_steps=props.sample_steps)
            for i, poly in enumerate(polys):
                # objeto temporal con atributos mínimos
                job_like = type("JobLike", (), {})()
                job_like.comp = props.contour_generation_mode
                job_like.diameter = props.diameter
                job_like.cut_direction = 'COUNTERCLOCKWISE'

                # ⚡ Mantener puntos como pares (x,y) en coordenadas locales
                compensated = utils.compensated_poly(poly, job_like)

                name = f"{props.contour_generation_mode}_{obj.name}_{i}"
                new_obj = create_offset_curve_object(name, compensated, ref_obj=obj)

                # Material según modo de compensación
                rgb = color_map.get(props.contour_generation_mode, (1.0, 1.0, 1.0))
                mat_name = f"Mat_{props.contour_generation_mode}"
                mat = ensure_material_with_color(mat_name, rgb)

                new_obj.data.materials.clear()
                new_obj.data.materials.append(mat)

                created += 1

        self.report({'INFO'}, f"Contornos generados: {created}")
        return {'FINISHED'}
# OPERADOR: Añadir trabajo
class OT_AddJob(bpy.types.Operator):
    bl_idname = "cnc.add_job"
    bl_label = "Añadir corte"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.cnc_props
        sel = [o for o in context.selected_objects if o.type == 'CURVE']
        if not sel:
            self.report({'WARNING'}, "Selecciona al menos un objeto CURVE para crear corte")
            return {'CANCELLED'}

        names = ",".join([o.name for o in sel])
        job = props.jobs.add()
        job.name = f"corte_{len(props.jobs)}"
        job.object_names = names
        job.depth_final = props.depth_final
        job.depth_per_pass = props.depth_per_pass
        job.feed = props.feed
        job.plunge = props.plunge
        job.z_safe = props.z_safe
        job.sample_steps = props.sample_steps
        job.diameter = props.diameter
        job.comp = props.comp
        job.tabs_enable = False
        job.tab_length = 5.0
        job.tab_thickness = 2.0
        job.tab_count = 3
        props.jobs_index = len(props.jobs) - 1

        self.report({'INFO'}, f"Corte añadido con objetos: {names}")
        return {'FINISHED'}


# OPERADOR: Quitar trabajo
class OT_RemoveJob(bpy.types.Operator):
    bl_idname = "cnc.remove_job"
    bl_label = "Quitar corte"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return len(context.scene.cnc_props.jobs) > 0

    def execute(self, context):
        props = context.scene.cnc_props
        idx = props.jobs_index
        props.jobs.remove(idx)
        props.jobs_index = max(0, idx - 1)
        self.report({'INFO'}, "Corte eliminado")
        return {'FINISHED'}


# OPERADOR: Limpiar cola de trabajos
class OT_ClearJobs(bpy.types.Operator):
    bl_idname = "cnc.clear_jobs"
    bl_label = "Limpiar cola"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.cnc_props
        props.jobs.clear()
        props.jobs_index = 0
        self.report({'INFO'}, "Cola de trabajos limpiada")
        return {'FINISHED'}


# OPERADOR: Visualizar pestañas
class OT_VisualizeTabs(bpy.types.Operator):
    bl_idname = "cnc.visualize_tabs"
    bl_label = "Visualizar pestañas"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.cnc_props
        if len(props.jobs) == 0:
            self.report({'WARNING'}, "No hay cortes definidos")
            return {'CANCELLED'}

        job = props.jobs[props.jobs_index]

        if not job.tabs_enable:
            self.report({'INFO'}, "Las pestañas no están activadas en este corte")
            return {'CANCELLED'}

        sel = [o for o in context.selected_objects if o.type == 'CURVE']
        if not sel:
            self.report({'WARNING'}, "Selecciona al menos una curva")
            return {'CANCELLED'}

        # Crear colección para pestañas
        col_name = "CNC_Tabs"
        if col_name in bpy.data.collections:
            col = bpy.data.collections[col_name]
            for o in list(col.objects):
                col.objects.unlink(o)
        else:
            col = bpy.data.collections.new(col_name)
            context.scene.collection.children.link(col)

        created = 0
        for obj in sel:
            polys = utils.sample_curve_object_to_xy_list(obj, sample_steps=job.sample_steps)
            for poly in polys:
                dists = utils.point_distance_along_poly(poly)

                # AUTO
                if job.tab_mode == 'AUTO':
                    ranges = utils.compute_tab_ranges_on_poly(poly, job.tab_length, job.tab_count)
                    for (s, e) in ranges:
                        pts = []
                        for target in (s, e):
                            for j in range(1, len(dists)):
                                if dists[j-1] <= target <= dists[j]:
                                    t = (target - dists[j-1]) / (dists[j] - dists[j-1])
                                    x = poly[j-1][0] + t * (poly[j][0] - poly[j-1][0])
                                    y = poly[j-1][1] + t * (poly[j][1] - poly[j-1][1])
                                    pts.append((x, y))
                                    break
                        if len(pts) == 2:
                            self._create_tab_visual(col, job, pts, created)
                            created += 1

                # MANUAL
                elif job.tab_mode == 'MANUAL':
                    for tab in job.custom_tabs:
                        min_i = min(range(len(poly)), key=lambda i: (Vector(poly[i]) - Vector((tab.x, tab.y))).length)
                        dist = dists[min_i]
                        half = job.tab_length / 2.0
                        s, e = dist - half, dist + half

                        pts = []
                        for target in (s, e):
                            for j in range(1, len(dists)):
                                if dists[j-1] <= target <= dists[j]:
                                    t = (target - dists[j-1]) / (dists[j] - dists[j-1])
                                    x = poly[j-1][0] + t * (poly[j][0] - poly[j-1][0])
                                    y = poly[j-1][1] + t * (poly[j][1] - poly[j-1][1])
                                    pts.append((x, y))
                                    break
                        if len(pts) == 2:
                            self._create_tab_visual(col, job, pts, created)
                            created += 1

        self.report({'INFO'}, f"Pestañas visualizadas: {created}")
        return {'FINISHED'}

    def _create_tab_visual(self, col, job, pts, created):
        curve_data = bpy.data.curves.new(name=f"TabCurve_{created:03d}", type='CURVE')
        curve_data.dimensions = '2D'
        spline = curve_data.splines.new('POLY')
        spline.points.add(1)
        spline.points[0].co = (pts[0][0], pts[0][1], 0.0, 1.0)
        spline.points[1].co = (pts[1][0], pts[1][1], 0.0, 1.0)

        curve_data.bevel_depth = 0.3
        curve_data.bevel_resolution = 0

        obj_tab = bpy.data.objects.new(f"Tab_{job.name}_{created:03d}", curve_data)

        mat = bpy.data.materials.get("Mat_Tab")
        if mat is None:
            mat = bpy.data.materials.new(name="Mat_Tab")
            mat.diffuse_color = (1.0, 1.0, 0.0, 1.0)  # amarillo
        obj_tab.data.materials.append(mat)

        col.objects.link(obj_tab)

        tab_len = (Vector(pts[1]) - Vector(pts[0])).length
        txt_data = bpy.data.curves.new(name=f"TabText_{created:03d}", type='FONT')
        txt_data.body = f"{tab_len:.2f} mm"
        txt_obj = bpy.data.objects.new(f"TabText_{created:03d}", txt_data)
        txt_obj.location = ((pts[0][0] + pts[1][0]) / 2.0,
                            (pts[0][1] + pts[1][1]) / 2.0,
                            0.0)
        txt_obj.scale = (0.8, 0.8, 0.8)
        col.objects.link(txt_obj)

# OPERADOR: Exportar G-code
class OT_ExportJobs(bpy.types.Operator):
    bl_idname = "cnc.export_jobs"
    bl_label = "Exportar G-code"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.cnc_props
        if len(props.jobs) == 0:
            self.report({'WARNING'}, "No hay cortes definidos")
            return {'CANCELLED'}

        file_path = bpy.path.abspath(props.file_path)
        if not file_path:
            self.report({'WARNING'}, "Debes definir un archivo de salida")
            return {'CANCELLED'}

        gcode_lines = []
        gcode_lines.append(";(G-code generado por Blender - Addon corte CNC Popular)")
        gcode_lines.append("G21  ;(usar mm)")
        gcode_lines.append("G90  ;(coordenadas absolutas)")
        gcode_lines.append("G92 X0 Y0 Z0  ;establecer 0")
        gcode_lines.append(f"G0 Z{props.z_safe:.3f}")
        gcode_lines.append("M106  ;(encender router)")
        gcode_lines.append("G4 S4  ;(esperar max potencia)")


        for job in props.jobs:
            gcode_lines.append(f"(Job: {job.name})")

            # obtener curvas seleccionadas
            objs = [bpy.data.objects.get(n) for n in job.object_names.split(",") if n]
            objs = [o for o in objs if o and o.type == 'CURVE']

            for obj in objs:
                polys = utils.sample_curve_object_to_xy_list(obj, sample_steps=job.sample_steps)
                for poly in polys:
                    # calcular pestañas
                    per_path_tab_ranges = []
                    if job.tabs_enable:
                        if job.tab_mode == 'AUTO':
                            ranges = utils.compute_tab_ranges_on_poly(poly, job.tab_length, job.tab_count)
                            per_path_tab_ranges.append(ranges)
                        elif job.tab_mode == 'MANUAL':
                            dists = utils.point_distance_along_poly(poly)
                            ranges = []
                            for tab in job.custom_tabs:
                                # buscar punto más cercano en la polilínea
                                min_i = min(
                                    range(len(poly)),
                                    key=lambda i: (Vector(poly[i]) - Vector((tab.x, tab.y))).length
                                )
                                dist = dists[min_i]
                                half = job.tab_length / 2.0
                                ranges.append((dist - half, dist + half))
                            per_path_tab_ranges.append(ranges)

                    # aplicar compensación robusta antes de generar G-code
                    compensated = utils.compensated_poly(poly, job)

                    # generar G-code para este polígono compensado
                    gcode_lines.extend(
                        utils.generate_gcode_for_poly(
                            compensated,
                            job,
                            per_path_tab_ranges
                        )
                    )

        
        gcode_lines.append("M107 ; router OFF")
        gcode_lines.append("G0 X0 Y0 ; volver a 0\n")

        try:
            with open(file_path, "w", encoding="utf-8") as f:
                f.write("\n".join(gcode_lines))
        except Exception as e:
            self.report({'ERROR'}, f"No se pudo escribir el archivo: {e}")
            return {'CANCELLED'}

        self.report({'INFO'}, f"G-code exportado a {file_path}")
        return {'FINISHED'}

# OPERADOR: Añadir pestaña manual en selección
class OT_AddCustomTab(bpy.types.Operator):
    bl_idname = "cnc.add_custom_tab"
    bl_label = "Añadir pestaña en selección"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.cnc_props
        if len(props.jobs) == 0:
            self.report({'WARNING'}, "No hay trabajos definidos")
            return {'CANCELLED'}

        job = props.jobs[props.jobs_index]
        obj = context.active_object
        if not obj or obj.type != 'CURVE' or obj.mode != 'EDIT':
            self.report({'WARNING'}, "Debes estar en modo edición de una curva")
            return {'CANCELLED'}

        coords = []
        for spline in obj.data.splines:
            if spline.type == 'BEZIER':
                for bp in spline.bezier_points:
                    if bp.select_control_point:
                        coords.append((bp.co.x, bp.co.y))
            else:  # POLY o NURBS
                for pt in spline.points:
                    if pt.select:
                        coords.append((pt.co.x, pt.co.y))

        if not coords:
            self.report({'WARNING'}, "Selecciona al menos un punto de la curva")
            return {'CANCELLED'}

        for (x, y) in coords:
            tab = job.custom_tabs.add()
            tab.x = x
            tab.y = y

        self.report({'INFO'}, f"Pestañas añadidas: {len(coords)}")
        return {'FINISHED'}


# OPERADOR: Reiniciar pestañas manuales
class OT_ResetCustomTabs(bpy.types.Operator):
    bl_idname = "cnc.reset_custom_tabs"
    bl_label = "Reiniciar pestañas manuales"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.cnc_props
        if len(props.jobs) == 0:
            self.report({'WARNING'}, "No hay trabajos definidos")
            return {'CANCELLED'}

        job = props.jobs[props.jobs_index]
        count = len(job.custom_tabs)
        job.custom_tabs.clear()

        self.report({'INFO'}, f"Se reiniciaron {count} pestañas manuales")
        return {'FINISHED'}

class OT_VisualizeSelectedJob(bpy.types.Operator):
    bl_idname = "cnc.visualize_selected_job"
    bl_label = "Visualizar Job seleccionado"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.cnc_props
        if len(props.jobs) == 0:
            self.report({'WARNING'}, "No hay trabajos definidos")
            return {'CANCELLED'}

        job = props.jobs[props.jobs_index]

        # Crear o recuperar colección para este Job
        coll_name = f"Job_{job.name}_Visual"
        if coll_name in bpy.data.collections:
            coll = bpy.data.collections[coll_name]
        else:
            coll = bpy.data.collections.new(coll_name)
            bpy.context.scene.collection.children.link(coll)

        objs = [bpy.data.objects.get(n) for n in job.object_names.split(",") if n]
        objs = [o for o in objs if o and o.type == 'CURVE']

        created = 0
        for obj in objs:
            polys = utils.sample_curve_object_to_xy_list(obj, sample_steps=job.sample_steps)
            for i, poly in enumerate(polys):
                compensated = utils.compensated_poly(poly, job)

                name = f"Job_{job.name}_{job.comp}_{obj.name}_{i}"
                new_obj = utils.create_offset_curve_object(name, compensated)

                # Color según compensación
                color_map = {
                    'CENTER': (1.0, 1.0, 1.0, 1.0),
                    'Interior': (1.0, 0.2, 0.2, 1.0),
                    'Exterior': (0.2, 1.0, 0.2, 1.0)
                }
                mat = bpy.data.materials.new(name=f"Mat_{job.comp}")
                mat.diffuse_color = color_map.get(job.comp, (0.5, 0.5, 0.5, 1.0))
                new_obj.data.materials.append(mat)

                # Enlazar a la colección del Job
                coll.objects.link(new_obj)
                if new_obj.name in bpy.context.scene.collection.objects:
                    bpy.context.scene.collection.objects.unlink(new_obj)

                created += 1

        self.report({'INFO'}, f"Visualización creada: {created} curvas en colección '{coll_name}'")
        return {'FINISHED'}

# --- Operador para visualizar orden ---


class OT_VisualizeJobsOrder(bpy.types.Operator):
    bl_idname = "cnc.visualize_jobs_order"
    bl_label = "Visualizar orden de Jobs"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.cnc_props
        if len(props.jobs) == 0:
            self.report({'WARNING'}, "No hay trabajos definidos")
            return {'CANCELLED'}

        coll_name = "Jobs_Order_Visual"
        if coll_name in bpy.data.collections:
            coll = bpy.data.collections[coll_name]
        else:
            coll = bpy.data.collections.new(coll_name)
            bpy.context.scene.collection.children.link(coll)

        # Limpiar textos previos
        for obj in list(coll.objects):
            bpy.data.objects.remove(obj, do_unlink=True)

        # Crear textos nuevos
        for idx, job in enumerate(props.jobs):
            obj_names = job.object_names.split(",")
            if not obj_names:
                continue
            ref_obj = bpy.data.objects.get(obj_names[0])
            if not ref_obj:
                continue

            txt_data = bpy.data.curves.new(type="FONT", name=f"JobOrder_{job.name}")
            txt_data.body = str(idx + 1)
            txt_obj = bpy.data.objects.new(f"JobOrder_{job.name}", txt_data)

            # Colocar el número al costado del objeto (1 unidad en X)
            txt_obj.location = ref_obj.location + Vector((1.0, 0.0, 0.0))
            txt_obj.scale = (0.5, 0.5, 0.5)

            # Material amarillo para los números
            mat_name = f"Mat_Order_{job.name}"
            mat = bpy.data.materials.get(mat_name)
            if mat is None:
                mat = bpy.data.materials.new(name=mat_name)
            mat.use_nodes = True
            bsdf = mat.node_tree.nodes.get("Principled BSDF")
            if bsdf:
                bsdf.inputs["Base Color"].default_value = (1.0, 1.0, 0.2, 1.0)  # amarillo RGBA

            txt_obj.data.materials.clear()
            txt_obj.data.materials.append(mat)

            coll.objects.link(txt_obj)

        self.report({'INFO'}, "Orden de Jobs visualizado")
        return {'FINISHED'}


# --- Operadores de mover Job ---
class OT_MoveJobUp(bpy.types.Operator):
    bl_idname = "cnc.move_job_up"
    bl_label = "Mover Job arriba"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.cnc_props
        idx = props.jobs_index
        if idx > 0:
            props.jobs.move(idx, idx - 1)
            props.jobs_index -= 1
            # 🔥 Actualizar visualización
            bpy.ops.cnc.visualize_jobs_order()
            return {'FINISHED'}
        else:
            self.report({'WARNING'}, "El corte ya está en la primera posición")
            return {'CANCELLED'}


class OT_MoveJobDown(bpy.types.Operator):
    bl_idname = "cnc.move_job_down"
    bl_label = "Mover Job abajo"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.cnc_props
        idx = props.jobs_index
        if idx < len(props.jobs) - 1:
            props.jobs.move(idx, idx + 1)
            props.jobs_index += 1
            # 🔥 Actualizar visualización
            bpy.ops.cnc.visualize_jobs_order()
            return {'FINISHED'}
        else:
            self.report({'WARNING'}, "El corte ya está en la última posición")
            return {'CANCELLED'}





# -------------------------------
# REGISTRO DE CLASES
# -------------------------------
classes = (
    OT_SelectOutputPath,
    OT_AddJob,
    OT_RemoveJob,
    OT_ClearJobs,
    OT_ExportJobs,
    OT_VisualizeSelectedCompensation,
    OT_VisualizeTabs,
    OT_AddCustomTab,
    OT_ResetCustomTabs,
    OT_VisualizeSelectedJob,
    OT_MoveJobUp,
    OT_MoveJobDown,
    OT_VisualizeJobsOrder, 
    
)

